<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SiteSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SiteSettingController extends Controller
{
    public function index()
    {
        $settings = SiteSetting::getSettings();
        return view('admin.site-settings.index', compact('settings'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'site_name' => 'required|string|max:255',
            'site_tagline' => 'nullable|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'favicon' => 'nullable|image|mimes:ico,png|max:512',
            'hero_background' => 'nullable|image|mimes:jpeg,png,jpg|max:5120',
            'primary_color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'secondary_color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'success_color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'warning_color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'danger_color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'dark_color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'footer_text' => 'nullable|string',
            'facebook_url' => 'nullable|url',
            'twitter_url' => 'nullable|url',
            'linkedin_url' => 'nullable|url',
            'instagram_url' => 'nullable|url',
        ]);

        $settings = SiteSetting::firstOrNew(['id' => 1]);

        // Procesar logo
        if ($request->hasFile('logo')) {
            if ($settings->logo_path) {
                Storage::disk('public')->delete($settings->logo_path);
            }
            $settings->logo_path = $request->file('logo')->store('site', 'public');
        }

        // Procesar favicon
        if ($request->hasFile('favicon')) {
            if ($settings->favicon_path) {
                Storage::disk('public')->delete($settings->favicon_path);
            }
            $settings->favicon_path = $request->file('favicon')->store('site', 'public');
        }

        // Procesar imagen de fondo del hero
        if ($request->hasFile('hero_background')) {
            if ($settings->hero_background_image) {
                Storage::disk('public')->delete($settings->hero_background_image);
            }
            $settings->hero_background_image = $request->file('hero_background')->store('site', 'public');
        }

        // Procesar redes sociales
        $socialLinks = [
            'facebook' => $request->facebook_url,
            'twitter' => $request->twitter_url,
            'linkedin' => $request->linkedin_url,
            'instagram' => $request->instagram_url,
        ];

        $settings->fill([
            'site_name' => $request->site_name,
            'site_tagline' => $request->site_tagline,
            'primary_color' => $request->primary_color,
            'secondary_color' => $request->secondary_color,
            'success_color' => $request->success_color,
            'warning_color' => $request->warning_color,
            'danger_color' => $request->danger_color,
            'dark_color' => $request->dark_color,
            'footer_text' => $request->footer_text,
            'social_links' => array_filter($socialLinks),
        ]);

        $settings->save();

        return redirect()->route('admin.site-settings.index')
                        ->with('success', 'Configuración del sitio actualizada exitosamente.');
    }

    public function resetColors()
    {
        $settings = SiteSetting::firstOrNew(['id' => 1]);
        
        $settings->update([
            'primary_color' => '#007bff',
            'secondary_color' => '#6c757d',
            'success_color' => '#28a745',
            'warning_color' => '#ffc107',
            'danger_color' => '#dc3545',
            'dark_color' => '#343a40',
        ]);

        return redirect()->route('admin.site-settings.index')
                        ->with('success', 'Colores restablecidos a los valores por defecto.');
    }
}
