<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Project;
use App\Models\ProjectImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class ProjectController extends Controller
{
    public function index()
    {
        $projects = Project::with(['mainImage', 'images'])->latest()->paginate(10);
        return view('admin.projects.index', compact('projects'));
    }

    public function create()
    {
        return view('admin.projects.create');
    }

    public function store(Request $request)
    {
        Log::info('🚀 INICIANDO CREACIÓN DE PROYECTO', [
            'user' => auth()->user()->email,
            'title' => $request->title,
            'has_main_image' => $request->hasFile('main_image'),
            'has_gallery' => $request->hasFile('gallery_images'),
            'request_all' => $request->all(),
            'files_info' => $request->hasFile('main_image') ? [
                'main_image_name' => $request->file('main_image')->getClientOriginalName(),
                'main_image_size' => $request->file('main_image')->getSize(),
                'main_image_valid' => $request->file('main_image')->isValid(),
            ] : 'No main image'
        ]);

        // Validación más simple
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'main_image' => 'required|file|max:10240', // 10MB, sin restricción de tipo por ahora
            'gallery_images.*' => 'nullable|file|max:10240',
        ]);

        Log::info('✅ VALIDACIÓN PASADA');

        try {
            DB::beginTransaction();

            // Crear proyecto
            $project = Project::create([
                'title' => $request->title,
                'description' => $request->description,
                'is_active' => $request->has('is_active'),
            ]);

            Log::info('✅ PROYECTO CREADO EN BD', ['project_id' => $project->id]);

            // Procesar imagen principal con método simple
            if ($request->hasFile('main_image')) {
                $this->processMainImageSimple($project, $request->file('main_image'));
            }

            // Procesar galería
            if ($request->hasFile('gallery_images')) {
                $this->processGalleryImagesSimple($project, $request->file('gallery_images'));
            }

            DB::commit();

            Log::info('🎉 PROYECTO CREADO EXITOSAMENTE', [
                'project_id' => $project->id,
                'total_images' => $project->images()->count()
            ]);

            return redirect()->route('admin.projects.index')
                           ->with('success', 'Proyecto creado exitosamente.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('❌ ERROR CREANDO PROYECTO', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()->withInput()->with('error', 'Error al crear el proyecto: ' . $e->getMessage());
        }
    }

    public function show(Project $project)
    {
        $project->load(['mainImage', 'galleryImages']);
        return view('admin.projects.show', compact('project'));
    }

    public function edit(Project $project)
    {
        $project->load(['mainImage', 'galleryImages']);
        return view('admin.projects.edit', compact('project'));
    }

    public function update(Request $request, Project $project)
    {
        Log::info('🔄 ACTUALIZANDO PROYECTO', [
            'project_id' => $project->id,
            'user' => auth()->user()->email,
            'has_main_image' => $request->hasFile('main_image'),
            'has_gallery' => $request->hasFile('gallery_images')
        ]);

        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'main_image' => 'nullable|file|max:10240',
            'gallery_images.*' => 'nullable|file|max:10240',
        ]);

        try {
            DB::beginTransaction();

            // Actualizar datos del proyecto
            $project->update([
                'title' => $request->title,
                'description' => $request->description,
                'is_active' => $request->has('is_active'),
            ]);

            Log::info('✅ DATOS ACTUALIZADOS', ['project_id' => $project->id]);

            // Procesar nueva imagen principal si se envió
            if ($request->hasFile('main_image')) {
                $this->processMainImageSimple($project, $request->file('main_image'));
            }

            // Procesar nuevas imágenes de galería
            if ($request->hasFile('gallery_images')) {
                $this->processGalleryImagesSimple($project, $request->file('gallery_images'));
            }

            DB::commit();

            Log::info('🎉 PROYECTO ACTUALIZADO EXITOSAMENTE', [
                'project_id' => $project->id,
                'total_images' => $project->images()->count()
            ]);

            return redirect()->route('admin.projects.edit', $project)
                           ->with('success', 'Proyecto actualizado exitosamente.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('❌ ERROR ACTUALIZANDO PROYECTO', [
                'project_id' => $project->id,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            return back()->withInput()->with('error', 'Error al actualizar el proyecto: ' . $e->getMessage());
        }
    }

    public function destroy(Project $project)
    {
        try {
            Log::info('🗑️ ELIMINANDO PROYECTO', ['project_id' => $project->id]);

            // Eliminar directorio completo
            $projectDir = "projects/proyecto-{$project->id}";
            if (Storage::disk('public')->exists($projectDir)) {
                Storage::disk('public')->deleteDirectory($projectDir);
            }

            $project->delete();

            Log::info('✅ PROYECTO ELIMINADO', ['project_id' => $project->id]);

            return redirect()->route('admin.projects.index')
                           ->with('success', 'Proyecto eliminado exitosamente.');

        } catch (\Exception $e) {
            Log::error('❌ ERROR ELIMINANDO PROYECTO', [
                'project_id' => $project->id,
                'error' => $e->getMessage()
            ]);
            
            return back()->with('error', 'Error al eliminar el proyecto: ' . $e->getMessage());
        }
    }

    public function deleteImage($imageId)
    {
        try {
            $image = ProjectImage::findOrFail($imageId);
            $project = $image->project;
            
            Log::info('🗑️ ELIMINANDO IMAGEN', [
                'image_id' => $imageId,
                'project_id' => $project->id,
                'is_main' => $image->is_main
            ]);

            // No permitir eliminar la imagen principal si es la única
            if ($image->is_main && $project->images()->count() === 1) {
                return response()->json([
                    'success' => false, 
                    'message' => 'No puedes eliminar la única imagen del proyecto.'
                ], 400);
            }

            $image->delete();

            // Si eliminamos la imagen principal, hacer principal la primera de galería
            if ($image->is_main) {
                $firstGalleryImage = $project->galleryImages()->first();
                if ($firstGalleryImage) {
                    $firstGalleryImage->update(['is_main' => true]);
                    Log::info('🖼️ NUEVA IMAGEN PRINCIPAL', ['image_id' => $firstGalleryImage->id]);
                }
            }

            Log::info('✅ IMAGEN ELIMINADA', ['image_id' => $imageId]);

            return response()->json(['success' => true, 'message' => 'Imagen eliminada correctamente']);

        } catch (\Exception $e) {
            Log::error('❌ ERROR ELIMINANDO IMAGEN', [
                'image_id' => $imageId,
                'error' => $e->getMessage()
            ]);
            return response()->json(['success' => false, 'message' => 'Error al eliminar la imagen'], 500);
        }
    }

    public function setMainImage($imageId)
    {
        try {
            $image = ProjectImage::findOrFail($imageId);
            $project = $image->project;

            Log::info('🖼️ CAMBIANDO IMAGEN PRINCIPAL', [
                'image_id' => $imageId,
                'project_id' => $project->id
            ]);

            // Quitar el estado principal de todas las imágenes
            $project->images()->update(['is_main' => false]);

            // Establecer la nueva imagen principal
            $image->update(['is_main' => true]);

            Log::info('✅ IMAGEN PRINCIPAL CAMBIADA', ['image_id' => $imageId]);

            return response()->json(['success' => true, 'message' => 'Imagen principal actualizada']);

        } catch (\Exception $e) {
            Log::error('❌ ERROR CAMBIANDO IMAGEN PRINCIPAL', [
                'image_id' => $imageId,
                'error' => $e->getMessage()
            ]);
            return response()->json(['success' => false, 'message' => 'Error al cambiar imagen principal'], 500);
        }
    }

    // MÉTODO SIMPLIFICADO - Igual que debe funcionar el logo
    private function processMainImageSimple(Project $project, $file)
    {
        Log::info('📸 PROCESANDO IMAGEN PRINCIPAL SIMPLE', [
            'project_id' => $project->id,
            'filename' => $file->getClientOriginalName(),
            'size' => $file->getSize(),
            'temp_path' => $file->getPathname(),
            'is_valid' => $file->isValid(),
            'error' => $file->getError()
        ]);

        try {
            // Eliminar imagen principal anterior si existe
            $oldMainImage = $project->mainImage;
            if ($oldMainImage) {
                $oldMainImage->delete();
                Log::info('🗑️ IMAGEN PRINCIPAL ANTERIOR ELIMINADA');
            }

            // Generar nombre simple
            $extension = $file->getClientOriginalExtension();
            $filename = 'main-' . time() . '.' . $extension;
            
            // Usar el método más simple posible - igual que el logo
            $path = $file->storeAs('projects', $filename, 'public');
            
            Log::info('💾 ARCHIVO GUARDADO', [
                'stored_path' => $path,
                'full_path' => storage_path('app/public/' . $path),
                'exists' => Storage::disk('public')->exists($path)
            ]);

            // Crear registro en BD
            $projectImage = ProjectImage::create([
                'project_id' => $project->id,
                'image_path' => $path,
                'original_name' => $file->getClientOriginalName(),
                'is_main' => true,
                'order' => 0,
                'file_size' => $file->getSize(),
            ]);

            Log::info('✅ IMAGEN PRINCIPAL PROCESADA', [
                'image_id' => $projectImage->id,
                'path' => $path
            ]);

        } catch (\Exception $e) {
            Log::error('❌ ERROR PROCESANDO IMAGEN PRINCIPAL', [
                'project_id' => $project->id,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            throw $e;
        }
    }

    private function processGalleryImagesSimple(Project $project, array $files)
    {
        Log::info('🖼️ PROCESANDO GALERÍA SIMPLE', [
            'project_id' => $project->id,
            'images_count' => count($files)
        ]);

        $currentMaxOrder = $project->galleryImages()->max('order') ?? 0;

        foreach ($files as $index => $file) {
            try {
                $extension = $file->getClientOriginalExtension();
                $filename = 'gallery-' . time() . '-' . $index . '.' . $extension;
                
                // Usar el método más simple posible
                $path = $file->storeAs('projects', $filename, 'public');

                // Crear registro en BD
                ProjectImage::create([
                    'project_id' => $project->id,
                    'image_path' => $path,
                    'original_name' => $file->getClientOriginalName(),
                    'is_main' => false,
                    'order' => $currentMaxOrder + $index + 1,
                    'file_size' => $file->getSize(),
                ]);

                Log::info('✅ IMAGEN DE GALERÍA PROCESADA', [
                    'index' => $index,
                    'filename' => $filename,
                    'path' => $path
                ]);

            } catch (\Exception $e) {
                Log::error('❌ ERROR PROCESANDO IMAGEN DE GALERÍA', [
                    'index' => $index,
                    'filename' => $file->getClientOriginalName(),
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    // MÉTODO DE DEBUG PARA VERIFICAR PERMISOS
    public function debugPermissions()
    {
        $info = [
            'storage_paths' => [
                'storage_app' => storage_path('app'),
                'storage_public' => storage_path('app/public'),
                'storage_projects' => storage_path('app/public/projects'),
            ],
            'public_paths' => [
                'public_storage' => public_path('storage'),
            ],
            'permissions' => [],
            'symlink_info' => [
                'exists' => is_link(public_path('storage')),
                'target' => is_link(public_path('storage')) ? readlink(public_path('storage')) : null,
            ],
            'php_info' => [
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
                'max_file_uploads' => ini_get('max_file_uploads'),
            ]
        ];

        // Verificar permisos
        foreach ($info['storage_paths'] as $key => $path) {
            $info['permissions'][$key] = [
                'path' => $path,
                'exists' => file_exists($path),
                'readable' => is_readable($path),
                'writable' => is_writable($path),
                'permissions' => file_exists($path) ? substr(sprintf('%o', fileperms($path)), -4) : null,
            ];
        }

        return response()->json($info, 200, [], JSON_PRETTY_PRINT);
    }

    // MÉTODO PARA CREAR DIRECTORIOS Y PERMISOS
    public function setupDirectories()
    {
        try {
            $directories = [
                storage_path('app'),
                storage_path('app/public'),
                storage_path('app/public/projects'),
            ];

            $results = [];

            foreach ($directories as $dir) {
                if (!file_exists($dir)) {
                    mkdir($dir, 0755, true);
                    $results[] = "Creado: $dir";
                } else {
                    $results[] = "Ya existe: $dir";
                }

                if (!is_writable($dir)) {
                    chmod($dir, 0755);
                    $results[] = "Permisos establecidos: $dir";
                }
            }

            // Crear symlink si no existe
            if (!is_link(public_path('storage'))) {
                \Artisan::call('storage:link');
                $results[] = "Symlink creado";
            }

            return response()->json([
                'success' => true,
                'results' => $results
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }
}
