<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SiteConfig;
use App\Models\Page;
use App\Models\Section;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class CmsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    // Configuración del sitio
    public function siteConfig()
    {
        $configs = SiteConfig::orderBy('group')->orderBy('order')->get()->groupBy('group');
        return view('admin.cms.site-config', compact('configs'));
    }

    public function updateSiteConfig(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            if ($request->hasFile($key)) {
                // Manejar archivos (logos, imágenes)
                $file = $request->file($key);
                $filename = time() . '_' . $file->getClientOriginalName();
                $path = $file->storeAs('public/cms', $filename);
                $value = 'storage/cms/' . $filename;
            }
            
            SiteConfig::set($key, $value);
        }

        SiteConfig::clearCache();

        return redirect()->back()->with('success', 'Configuración actualizada exitosamente.');
    }

    // Gestión de páginas
    public function pages()
    {
        $pages = Page::orderBy('menu_order')->get();
        return view('admin.cms.pages.index', compact('pages'));
    }

    public function createPage()
    {
        return view('admin.cms.pages.create');
    }

    public function storePage(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string|max:255',
        ]);

        $page = Page::create([
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'content' => $request->content,
            'meta_description' => $request->meta_description,
            'meta_keywords' => $request->meta_keywords,
            'is_active' => $request->has('is_active'),
            'show_in_menu' => $request->has('show_in_menu'),
            'menu_order' => $request->menu_order ?? 0,
        ]);

        return redirect()->route('admin.cms.pages.index')
                        ->with('success', 'Página creada exitosamente.');
    }

    public function editPage(Page $page)
    {
        return view('admin.cms.pages.edit', compact('page'));
    }

    public function updatePage(Request $request, Page $page)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string|max:255',
        ]);

        $page->update([
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'content' => $request->content,
            'meta_description' => $request->meta_description,
            'meta_keywords' => $request->meta_keywords,
            'is_active' => $request->has('is_active'),
            'show_in_menu' => $request->has('show_in_menu'),
            'menu_order' => $request->menu_order ?? 0,
        ]);

        return redirect()->route('admin.cms.pages.index')
                        ->with('success', 'Página actualizada exitosamente.');
    }

    public function destroyPage(Page $page)
    {
        $page->delete();
        return redirect()->route('admin.cms.pages.index')
                        ->with('success', 'Página eliminada exitosamente.');
    }

    // Gestión de secciones
    public function sections()
    {
        $sections = Section::orderBy('order')->get();
        return view('admin.cms.sections.index', compact('sections'));
    }

    public function editSection(Section $section)
    {
        return view('admin.cms.sections.edit', compact('section'));
    }

    public function updateSection(Request $request, Section $section)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required',
        ]);

        $data = [
            'title' => $request->title,
            'content' => $request->content,
            'is_active' => $request->has('is_active'),
            'order' => $request->order ?? 0,
        ];

        if ($request->hasFile('image')) {
            // Eliminar imagen anterior si existe
            if ($section->image && Storage::exists('public/' . str_replace('storage/', '', $section->image))) {
                Storage::delete('public/' . str_replace('storage/', '', $section->image));
            }

            $file = $request->file('image');
            $filename = time() . '_' . $file->getClientOriginalName();
            $path = $file->storeAs('public/cms/sections', $filename);
            $data['image'] = 'storage/cms/sections/' . $filename;
        }

        // Manejar configuraciones adicionales
        $settings = [];
        if ($request->has('button_text')) {
            $settings['button_text'] = $request->button_text;
        }
        if ($request->has('button_url')) {
            $settings['button_url'] = $request->button_url;
        }
        if ($request->has('background_color')) {
            $settings['background_color'] = $request->background_color;
        }
        
        $data['settings'] = $settings;

        $section->update($data);

        return redirect()->route('admin.cms.sections.index')
                        ->with('success', 'Sección actualizada exitosamente.');
    }

    // Limpiar caché
    public function clearCache()
    {
        SiteConfig::clearCache();
        
        return redirect()->back()->with('success', 'Caché limpiado exitosamente.');
    }
}
